<?php

/*********************************************************************
 *
 * tokengrid PDF class :: Extend the tokengrid class with PDF output
 *
 * Creation of PDF token grids using TCPDF.
 *
 * Any feedback is welcome !
 *
 *
 * LICENCE
 *
 *   Copyright (c) 2008, SysCo systemes de communication sa
 *   SysCo (tm) is a trademark of SysCo systemes de communication sa
 *   (http://www.sysco.ch/)
 *   All rights reserved.
 * 
 *   This file is part of the tokengrid class
 *
 *   The tokengrid class is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public License as
 *   published by the Free Software Foundation, either version 3 of the License,
 *   or (at your option) any later version.
 * 
 *   The tokengrid class is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 * 
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with the tokengrid class
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * @author: SysCo/al
 * @since CreationDate: 2008-04-22
 * @copyright (c) 2008 by SysCo systemes de communication sa
 * @version $LastChangedRevision: 1.1 $
 * @version $LastChangedDate: 2008-04-23s $
 * @version $LastChangedBy: SysCo/yj $
 * @link $HeadURL: tokengrid.pdf.class.php $
 * @link http://developer.sysco.ch/php/
 * @link developer@sysco.ch
 * Language: PHP 4.x or higher
 *
 *
 * Usage
 *
 *   require_once('tokengrid.pdf.class.php');
 *   $token_grid_pdf = new TokenGridPdf([$x_grid_size = 10[, $y_grid_size = 10[, $token_length = 4[, $grid_salt = 'T@kenGr!D']]]])
 *
 *
 * Examples (see the tokengrid.pdf.demo.php file for a working example)
 *
 *   Example 1 (create a grid)
 *     <?php
 *         require_once('tokengrid.pdf.class.php');
 *         $token_grid_pdf = new TokenGrid(10, 10, 4, "MySalt");
 *         header("Content-type: application/pdf");
 *         echo $token_grid_pdf->GetPdfGrid("SpecificUserId");
 *     ?>
 *
 *
 * External file needed
 *
 *   TCPDF LGPL library (http://www.tcpdf.org/)
 *
 *
 * Special issues
 *
 *   This class extension needs at least PHP5
 *
 *
 * Change Log
 *
 *   2008-04-23 1.2   SysCo/al code cleaning, layout changes
 *   2008-04-22 1.1   SysCo/yj PDF layout
 *   2008-04-21 1.0   SysCo/al Initial release
 *
 *********************************************************************/

define ("K_TCPDF_EXTERNAL_CONFIG", TRUE);

require_once('../tokengrid/tokengrid.class.php');
require_once('../tcpdf/tcpdf.php');


/*********************************************************************
 *
 * TokenGridPdf
 * Strong Authentication Token Grid, PDF extension
 *
 * Creation 2008-04-22
 * @package tokengrid
 * @version v.1.0
 * @author SysCo/al
 *
 *********************************************************************/
class TokenGridPdf extends TokenGrid
{
    private $_utf8_encoding_required = false;
    private $_display_token_grid_id = true;
    private $_display_token_grid_height = 2;
    private $_display_token_grid_font_size = 5;
    private $_grid_footer_txt = '';
    private $_frame_around_grid = false;

   /**
     * Informations concerning the pdf document
     */
    // Card width in mm
    private $_cardWidth  = 86.0;
    
    // Card height in mm
    private $_cardHeight = 54.0;
    
    private $_xCardPosition = 0;
    private $_yCardPosition = 0;
    
    
    /**
     * Informations concerning the pdf document
     */
    private $creator = "TokenGridPdf";
    private $author  = "TokenGridPdf";
    private $title   = "TokenGrid";
    private $subject = "TokenGrid";


    /**
     * Information concerning the header
     *
     * @param unknown_type $grid_id
     * @return unknown
     */
    private $headerLogo      = "";
    private $headerTitle     = "";
    private $headerString    = "";
    private $headerLogoWidth = 0;


    /*
     * Informations concerning card's header
     */
    private $cardHeaderBackColorH = array(255,255,255);
    private $cardHeaderFontColorH = array(0,0,0);
    private $cardHeaderFontSizeH = 10;
    private $cardHeaderFontH = "vera";
    private $cardHeaderLineWidthH = 0.3;
    private $cardHeaderFontPropertyH = '';
    private $cardHeaderLineColorH = array(0,0,0);
    
    private $cardHeaderBackColorV = array(255,255,255);
    private $cardHeaderFontColorV = array(0,0,0);
    private $cardHeaderFontSizeV = 10;
    private $cardHeaderFontV = "vera";
    private $cardHeaderLineWidthV = 0.3;
    private $cardHeaderFontPropertyV = '';
    private $cardHeaderLineColorV = array(0,0,0);
    
    private $cardCellsBackColor = array(255,255,255);
    private $cardCellsFontColor = array(0,0,0);
    private $cardCellsFontSize = 10;
    private $cardCellsFont = "vera";
    private $cardCellsLineWidth = 0.3;
    private $cardCellsFontProperty = '';
    private $cardCellsLineColor = array(0,0,0);
    
    
    /*
     * Informations about the front image of the card
     */
    private $cardFrontImageName="";
    private $cardFrontImageType="";
    private $imagePosition = "r"; //r,l,t,b    
    
    private $tabTextBox = array();


    public function TokenGridPdf($x_grid_size = 10, $y_grid_size = 10, $token_length = 4, $grid_salt = 'T@kenGr!D', $tcpdf_path_main = '', $tcpdf_path_images = '', $tcpdf_path_cache = '')
    {
        $this->TokenGrid($x_grid_size,$y_grid_size,$token_length,$grid_salt);

        define ("K_PATH_MAIN", ((''==trim($tcpdf_path_main))?str_replace("//","/",dirname(__FILE__)."/../tcpdf/"):$tcpdf_path_main));
        define ("K_PATH_IMAGES", ((''==trim($tcpdf_path_images))?K_PATH_MAIN."images/":$tcpdf_path_images));
        define ("K_PATH_CACHE", ((''==trim($tcpdf_path_cache))?K_PATH_MAIN."cache/":$tcpdf_path_cache));
        define ("K_BLANK_IMAGE", K_PATH_IMAGES."_blank.png");
        define ("K_PATH_FONTS", K_PATH_MAIN."fonts/");
        define ("PDF_HEADER_LOGO", "tcpdf_logo.jpg");
        define ("K_PATH_URL", "");
        define ("K_PATH_URL_CACHE", K_PATH_URL."cache/");
        define ("PDF_PAGE_FORMAT", "A4");
        define ("PDF_PAGE_ORIENTATION", "P");
        define ("PDF_CREATOR", "TCPDF");
        define ("PDF_AUTHOR", "TCPDF");
        define ("PDF_HEADER_TITLE", "tokengrid PDF class");
        define ("PDF_HEADER_STRING", "tokengrid PDF class");
        define ("PDF_HEADER_LOGO_WIDTH", 30);
        define ("PDF_UNIT", "mm");
        define ("PDF_MARGIN_HEADER", 5);
        define ("PDF_MARGIN_FOOTER", 10);
        define ("PDF_MARGIN_TOP", 27);
        define ("PDF_MARGIN_BOTTOM", 25);
        define ("PDF_MARGIN_LEFT", 15);
        define ("PDF_MARGIN_RIGHT", 15);
        define ("PDF_FONT_NAME_MAIN", "vera"); //vera
        define ("PDF_FONT_SIZE_MAIN", 10);
        define ("PDF_FONT_NAME_DATA", "vera"); //vera
        define ("PDF_FONT_SIZE_DATA", 8);
        define ("PDF_IMAGE_SCALE_RATIO", 4);
        define("HEAD_MAGNIFICATION", 1.1);
        define("K_CELL_HEIGHT_RATIO", 1.25);
        define("K_TITLE_MAGNIFICATION", 1.3);
        define("K_SMALL_RATIO", 2/3);
    }


    public function GetPdfGrid($grid_id, $grid_footer_txt = "")
    {
        $this->_grid_footer_txt = ('' != $grid_footer_txt)?$grid_footer_txt:$grid_id;

        // create new PDF document 
        $pdf = new TCPDF(PDF_PAGE_ORIENTATION, 'mm', PDF_PAGE_FORMAT, true); 

        // set document information 
        $pdf->SetCreator($this->creator); 
        $pdf->SetAuthor($this->author); 
        $pdf->SetTitle($this->title); 
        $pdf->SetSubject($this->subject);

        // Enable/disable default header and footer
        $pdf->setPrintHeader(true);
        $pdf->setPrintFooter(false);

        // set default header data 
        $pdf->SetHeaderData($this->headerLogo, $this->headerLogoWidth, $this->headerTitle, $this->headerString); 
        $pdf->setFooterFont(Array(PDF_FONT_NAME_DATA, '', PDF_FONT_SIZE_DATA));
 
        // set header and footer fonts 
        $pdf->setHeaderFont(Array(PDF_FONT_NAME_MAIN, '', PDF_FONT_SIZE_MAIN)); 
        $pdf->setFooterFont(Array(PDF_FONT_NAME_DATA, '', PDF_FONT_SIZE_DATA)); 

        //set margins 
        $pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT); 
        $pdf->SetHeaderMargin(PDF_MARGIN_HEADER); 
        $pdf->SetFooterMargin(PDF_MARGIN_FOOTER); 

        //set auto page breaks 
        $pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM); 

        //set image scale factor 
        $pdf->setImageScale(PDF_IMAGE_SCALE_RATIO); 

        //set some language-dependent strings 
        $pdf->setLanguageArray('en'); 

        //initialize document 
        $pdf->AliasNbPages(); 

        // add a page 
        $pdf->AddPage(); 
         
        $grid_array = $this->GetGridArray($grid_id);
        
        $tokenGridIdLineHeight = ($this->_display_token_grid_id)?$this->_display_token_grid_height:0;
        $gridHeight = $this->_cardHeight-$tokenGridIdLineHeight-$this->cardHeaderLineWidthH;
        
        // Calculcate the cell height and width
        $cellHeight = round(($gridHeight-$this->cardHeaderLineWidthH*($this->_y_grid_size))/($this->_y_grid_size+1),1);
        $cellWidth  = round(($this->_cardWidth-$this->cardHeaderLineWidthH*($this->_x_grid_size))/($this->_x_grid_size+1),1);

        // Goto the right position
        $top_align_offset = round(($this->_cardHeight-$tokenGridIdLineHeight-(($this->_y_grid_size+1)*$cellHeight)) / 2, 1);
        $left_align_offset = round(($this->_cardWidth-(($this->_x_grid_size+1)*$cellWidth)) / 2, 1);

        $pdf->SetXY($this->_xCardPosition+$left_align_offset,$this->_yCardPosition+$top_align_offset);
        $pdf->SetFillColor(0,0,0);
		$pdf->SetDrawColor($this->cardHeaderLineColorV[0],$this->cardHeaderLineColorV[1],$this->cardHeaderLineColorV[2]);
		$pdf->SetLineWidth(0.1);
		$pdf->SetFont("vera","",1);
        // Calcul la hauteur d'une cellule
        for ($line = 0; $line <= $this->_y_grid_size; $line++)
        {
            for ($col = 0; $col <= $this->_x_grid_size; $col++)
            {
                if (0 == $col)
                {
                    if (0 == $line)
                    {
                        $pdf->SetFillColor($this->cardHeaderBackColorV[0],$this->cardHeaderBackColorV[1],$this->cardHeaderBackColorV[2]);
				        $pdf->SetTextColor($this->cardHeaderFontColorV[0],$this->cardHeaderFontColorV[1],$this->cardHeaderFontColorV[2]);
				        $pdf->SetDrawColor($this->cardHeaderLineColorV[0],$this->cardHeaderLineColorV[1],$this->cardHeaderLineColorV[2]);
				        $pdf->SetLineWidth($this->cardHeaderLineWidthV);
				        $pdf->SetFont($this->cardHeaderFontV,$this->cardHeaderFontPropertyV, $this->cardHeaderFontSizeV); 
                        $pdf->Cell($cellWidth,$cellHeight,'',1,0,'C',1);
                    }
                    else
                    {
                        $pdf->SetFillColor($this->cardHeaderBackColorV[0],$this->cardHeaderBackColorV[1],$this->cardHeaderBackColorV[2]);
				        $pdf->SetTextColor($this->cardHeaderFontColorV[0],$this->cardHeaderFontColorV[1],$this->cardHeaderFontColorV[2]);
				        $pdf->SetDrawColor($this->cardHeaderLineColorV[0],$this->cardHeaderLineColorV[1],$this->cardHeaderLineColorV[2]);
				        $pdf->SetLineWidth($this->cardHeaderLineWidthV);
				        $pdf->SetFont($this->cardHeaderFontV,$this->cardHeaderFontPropertyV, $this->cardHeaderFontSizeV); 
                        $pdf->Cell($cellWidth,$cellHeight,$line,1,0,'C',1);
                    }
                }
                else // (0 != $col)
                {
                    if (0 == $line)
                    {
                        $pdf->SetFillColor($this->cardHeaderBackColorH[0],$this->cardHeaderBackColorH[1],$this->cardHeaderBackColorH[2]);
				        $pdf->SetTextColor($this->cardHeaderFontColorH[0],$this->cardHeaderFontColorH[1],$this->cardHeaderFontColorH[2]);
				        $pdf->SetDrawColor($this->cardHeaderLineColorH[0],$this->cardHeaderLineColorH[1],$this->cardHeaderLineColorH[2]);
				        $pdf->SetLineWidth($this->cardHeaderLineWidthH);
				        $pdf->SetFont($this->cardHeaderFontH,$this->cardHeaderFontPropertyH, $this->cardHeaderFontSizeH); 
				        $pdf->Cell($cellWidth,$cellHeight,substr($this->_horizontal_grid_labels, $col - 1, 1),1,0,'C',1);
                    }
                    else
                    {
                       $pdf->SetFillColor($this->cardCellsBackColor[0],$this->cardCellsBackColor[1],$this->cardCellsBackColor[2]);
				       $pdf->SetTextColor($this->cardCellsFontColor[0],$this->cardCellsFontColor[1],$this->cardCellsFontColor[2]);
				       $pdf->SetDrawColor($this->cardCellsLineColor[0],$this->cardCellsLineColor[1],$this->cardCellsLineColor[2]);
				       $pdf->SetLineWidth($this->cardCellsLineWidth);
				       $pdf->SetFont($this->cardCellsFont,$this->cardCellsFont, $this->cardCellsFontSize);
                       $pdf->Cell($cellWidth,$cellHeight,$grid_array[$line][$col],1,0,'C',1);
                    }
                }
            }
            $pdf->Ln();
            $pdf->SetX($this->_xCardPosition+$left_align_offset);
        }
        if ($this->_display_token_grid_id)
        {
            $pdf->SetFillColor(255,255,255);
            $pdf->SetTextColor($this->cardCellsFontColor[0],$this->cardCellsFontColor[1],$this->cardCellsFontColor[2]);
            $pdf->SetDrawColor($this->cardCellsLineColor[0],$this->cardCellsLineColor[1],$this->cardCellsLineColor[2]);
            $pdf->SetLineWidth($this->cardCellsLineWidth);
            $pdf->SetFont($this->cardHeaderFontV,$this->cardHeaderFontPropertyV, $this->_display_token_grid_font_size); 
            $pdf->Cell($cellWidth*($this->_x_grid_size+1),$this->_display_token_grid_height,$this->_grid_footer_txt,1,0,'R',1);
        }


        /**
         * Show the card image
         */
        if(file_exists($this->cardFrontImageName))
        {
           $posXImage;
           $posYImage;
           
           switch($this->imagePosition)
           {
              case 'l' : $posXImage = $this->_xCardPosition-$this->_cardWidth;
                         $posYImage = $this->_yCardPosition; break;
                         
              case 't' : $posXImage = $this->_xCardPosition;
                         $posYImage = $this->_yCardPosition-$this->_cardHeight; break;
                         
              case 'b' : $posXImage = $this->_xCardPosition;
                         $posYImage = $this->_yCardPosition+$this->_cardHeight; break;
              case 'r' : 
              default : $posXImage = $this->_xCardPosition+$this->_cardWidth;
                         $posYImage = $this->_yCardPosition; break;
           }
           $pdf->Image($this->cardFrontImageName,$posXImage,$posYImage,$this->_cardWidth, $this->_cardHeight, $this->cardFrontImageType);
        }
        
        // Show textBoxs
        for($i=0; $i< count($this->tabTextBox); $i++)
        {
           $this->tabTextBox[$i]->printTextZone($pdf);
        }


        if ($this->_frame_around_grid)
        {
            $pdf->Rect($this->_xCardPosition,$this->_yCardPosition, $this->_cardWidth, $this->_cardHeight, 'D');
        }

        if ($this->_frame_around_grid)
        {
            $pdf->Rect($posXImage,$posYImage, $this->_cardWidth, $this->_cardHeight, 'D');
        }

        //Close and output PDF document 
        return $pdf->Output("","S"); 
    }


 	/**
    * Require UTF8 encoding for the different string passe in parameters
    *
    * @param boolean $utf8_encoding_required
    */
    public function SetUtf8EncodingRequired($utf8_encoding_required = false)
    {
        $this->_utf8_encoding_required = $utf8_encoding_required;
    }


 	/**
    * Set if the token grid id is displayed on the credit card sized paper or not
    *
    * @param boolean
    */
    public function SetDisplayTokenGridId($display_token_grid_id = true)
    {
        $this->_display_token_grid_id = $display_token_grid_id;
    }


 	/**
    * Set if we need to display a frame around the grid
    *
    * @param boolean
    */
    public function SetFrameAroundGrid($frame_around_grid = true)
    {
        $this->_frame_around_grid = $frame_around_grid;
    }


    /**
    * Encode a string if the user ask for
    *
    * @param string $string
    * @return string
    */
    private function Utf8EncodeIfNeeded($string = "")
    {
      return (($this->_utf8_encoding_required)?utf8_encode($string):$string);
    }


    /**
     * Set the card top-left position
     *
     * @param int $x X position in mm
     * @param int $y Y position in mm
     */
    function setCardPosition($x=0,$y=0)
    {
       $this->_xCardPosition = $x;
       $this->_yCardPosition = $y;
    }
    
    /**
     * Set the pdf document properties
     *
     * @param string $creator Document's creator
     * @param string $author  Document's author
     * @param string $title	  Document's title
     * @param string $subject Document's subject
     */
    function setDocumentProperties($creator="", $author="", $title="", $subject="")
    {
	    $this->creator = $this->Utf8EncodeIfNeeded($creator);
	    $this->author  = $this->Utf8EncodeIfNeeded($author);
	    $this->title   = $this->Utf8EncodeIfNeeded($title);
	    $this->subject = $this->Utf8EncodeIfNeeded($subject);
    }
    
    /**
     * Allow the user to setup the header
     *
     * @param string $logo
     * @param int $logowidth
     * @param string $headerString
     * @param string $headerTitle
     */
    function setHeader($logo="", $logowidth=0, $headerString="", $headerTitle="")
    {
       $this->headerLogo      = $this->Utf8EncodeIfNeeded($logo);
       $this->headerTitle     = $this->Utf8EncodeIfNeeded($headerTitle);
       $this->headerString    = $this->Utf8EncodeIfNeeded($headerString);
       $this->headerLogoWidth = $logowidth;
    }
    
    /**
     * Setup the size of the card
     *
     * @param int $width  Card width in userUnit
     * @param int $height Card height in userUnit
     */
    function setCardSize ($width=86, $height=54)
    {
       $this->_cardWidth  = $width;
       $this->_cardHeight = $height;
    }
    
    /**
     * Setup the style of the horizontal header
     *
     * @param int array[3] $backColor Background colour (#rgb)
     * @param int array[3] $fontColor Font colour (#rgb)
     * @param int $fontSize Font's size
     * @param string $font Font's name
     * @param string $fontProperty Font style (empty string: regular, B: bold, I: italic, U: underline, D: line trough)
     * @param float $lineWidth Line Width
     * @param int array[3] $lineColor Line's colour
     */
    function setStyleHeaderHonrizontal($backColor=array(255,255,255), $fontColor=array(0,0,0), $fontSize=10, $font="vera", $fontProperty="", $lineWidth=0.3, $lineColor=array(0,0,0))
    {
      $this->cardHeaderBackColorH    = $backColor;
      $this->cardHeaderFontColorH    = $fontColor;
      $this->cardHeaderFontSizeH     = $fontSize;
      $this->cardHeaderFontH         = $font;
      $this->cardHeaderFontPropertyH = $fontProperty;
      $this->cardHeaderLineWidthH    = $lineWidth;
      $this->cardHeaderLineColorH     = $lineColor;
    }
    
    /**
     * Setup the style of the vetical header
     *
     * @param int array[3] $backColor Background colour (#rgb)
     * @param int array[3] $fontColor Font colour (#rgb)
     * @param int $fontSize Font's size
     * @param string $font Font's name
     * @param string $fontProperty Font style (empty string: regular, B: bold, I: italic, U: underline, D: line trough)
     * @param float $lineWidth Line Width
     * @param int array[3] $lineColor Line's colour
     */
    function setStyleHeaderVertical($backColor=array(255,255,255), $fontColor=array(0,0,0), $fontSize=10, $font="vera", $fontProperty="", $lineWidth=0.3, $lineColor=array(0,0,0))
    {
      $this->cardHeaderBackColorV    = $backColor;
      $this->cardHeaderFontColorV    = $fontColor;
      $this->cardHeaderFontSizeV    = $fontSize;
      $this->cardHeaderFontV         = $font;
      $this->cardHeaderFontPropertyV = $fontProperty;
      $this->cardHeaderLineWidthV    = $lineWidth;
      $this->cardHeaderLineColorV     = $lineColor;
    }
    
    /**
     * Setup the style of the cells
     *
     * @param int array[3] $backColor Background colour (#rgb)
     * @param int array[3] $fontColor Font colour (#rgb)
     * @param int $fontSize Font's size
     * @param string $font Font's name
     * @param string $fontProperty Font style (empty string: regular, B: bold, I: italic, U: underline, D: line trough)
     * @param float $lineWidth Line Width
     * @param int array[3] $lineColor Line's colour
     */
    function setStyleCells($backColor=array(255,255,255), $fontColor=array(0,0,0), $fontSize=10, $font="vera", $fontProperty="", $lineWidth=0.3, $lineColor=array(0,0,0))
    {
      $this->cardCellsBackColor    = $backColor;
      $this->cardCellsFontColor    = $fontColor;
      $this->cardCellsFontSize     = $fontSize;
      $this->cardCellsFont         = $font;
      $this->cardCellsFontProperty = $fontProperty;
      $this->cardCellsLineWidth    = $lineWidth;
      $this->cardCellsLineColor    = $lineColor;
    }
    
    /**
     * Set the image next to the card
     *
     * @param string $fileName Image's name
     * @param string $type Image's type (jpg, png)
     */
    function setImageFrontCard($fileName="", $type="")
    {
       $this->cardFrontImageName = $fileName;
       $this->cardFrontImageType = $type;
    }

    /**
     * Setup the side where to put the image
     *
     * @param char $side r: right, l:left, t:top, b:bottom
     */
    function setImageFrontPosition($side='r')
    {
       switch($side)
       {
          case 'r':
          case 'l':
          case 'b':
          case 't': $this->imagePosition = $side; break;
          default : $this->imagePosition = 'r';
       }       
    }

    /**
     * Allow the user to add a textBox
     *
     * @param int $width
     * @param int $height
     * @param string $text
     * @param string $border
     * @param int $ln
     * @param char $align
     * @param int $fill
     * @param int array[3] $backgourndColor
     * @param mixed $link
     * @param int $stretch
     * @param int $PosX
     * @param int $PosY
     * @return unknown
     */
    function addTextZone($width=0, $height=0, $text="", $border=0, $align="L", $fill=0, $ln=1, $PosX=0, $PosY=0, $resetH=false, $stretch = 0, $isHtml = false, $backGroundColor= array(255,255,255))
    {
       $this->tabTextBox[]= new zoneText($width, $height, $this->Utf8EncodeIfNeeded($text), $border, $align, $fill, $ln, $PosX, $PosY, $resetH, $stretch, $isHtml, $backGroundColor);
       return count($this->tabTextBox)-1;
    }
    
    /**
     * Permet de rcuprer une zone de texte afin de lui modifier les attributs
     *
     * @param unknown_type $indice
     * @return unknown
     */
    function getTextZone($indice)
    {
       return $this->tabTextBox[$indice];
    }
}

class zoneText
{
   private $width=0;
   private $height=0;
   private $text = "";
   private $border = "";
   private $borderWidth = 0.3;
   private $borderColor = array(0,0,0);
   private $ln=0;
   private $align="L";
   private $fill=0;
   private $link="";
   private $stretch =0;
   private $PosX=0;
   private $PosY=0;
   private $backGroundColor= array(255,255,255);
   private $font= "vera";
   private $fontColor = array(0,0,0);
   private $fontSize = 15;
   private $fontProperties = ""; 
   private $resetH = false;
   private $isHtml = false;

   /**
    * Create a new textBox
    *
    * @param int $width
    * @param int $height
    * @param string $text
    * @param string $border
    * @param int $ln
    * @param char $align
    * @param int $fill
    * @param int array[3] $backgourndColor
    * @param mixed $link
    * @param int $stretch
    * @param int $PosX
    * @param int $PosY
    * @return zoneText
    */
   public function zoneText($width=0, $height=0, $text="", $border="", $align="L", $fill=0, $ln=1, $PosX=0, $PosY=0, $resetH=false, $stretch = 0, $isHtml = false, $backGroundColor= array(255,255,255))
   {
       $this->width=$width;
	   $this->height=$height;
	   $this->text = $text;
	   $this->border = $border;
	   $this->ln=$ln;
	   $this->align=$align;
	   $this->fill=$fill;
	   $this->stretch =$stretch;
	   $this->PosX=$PosX;
	   $this->PosY=$PosY;
	   $this->backGroundColor= $backGroundColor;
	   $this->resetH = $resetH;
	   $this->isHtml = $isHtml;
   }
   
   /**
    * Setup the font style
    *
    * @param srting $name Font name
    * @param int $size Font size
    * @param int array[3] $color Font color
    * @param string $properties Font style
    */
   public function setFont($name="vera", $size=15, $color=array(0,0,0), $properties="")
   {
      $this->font           = $name;
      $this->fontSize       = $size;
      $this->fontColor      = $color;
      $this->fontProperties = $properties;
   }
   
   /**
    * Allow the user to setup the border of one textBox
    *
    * @param int array[3] $color Line color
    * @param int $width Width of the line
    */
   public function setBorder($color=array(0,0,0), $width=0.3)
   {
      $this->borderColor = $color;
      $this->borderWidth = $width;
   }
  
   /**
    * Print the box on the pdf document
    *
    * @param TCPDF $pdf PDF document
    */
   public function printTextZone($pdf)
   {
      $pdf->SetFillColor($this->backGroundColor[0],$this->backGroundColor[1],$this->backGroundColor[2]);
      $pdf->SetTextColor($this->fontColor[0],$this->fontColor[1],$this->fontColor[2]);
      $pdf->SetDrawColor($this->borderColor[0],$this->borderColor[1],$this->borderColor[2]);
      $pdf->SetLineWidth($this->borderWidth);
      $pdf->SetFont($this->font,$this->fontProperties, $this->fontSize);
      $pdf->MultiCell ($this->width, $this->height, $this->text, $this->border, $this->align, $this->fill, $this->ln, $this->PosX, $this->PosY, $this->resetH, $this->stretch, $this->isHtml);
   }
}


?>